// NOTE: Follow the security guide while implementing plugin app https://www.electronjs.org/docs/tutorial/security
const {app, BrowserWindow, ipcMain, shell, dialog, Menu} = require('electron')
const path = require('path');

const PLUGIN_ID = 'com.krock.io.extension'

if (process.platform === "darwin") {
    WorkflowIntegration = require(path.join(__dirname, "WorkflowIntegration_Mac.node"));
} else {
    WorkflowIntegration = require(path.join(__dirname, 'WorkflowIntegration_Win.node'));
}

// Cached objects
let resolveObj = null;
let projectManagerObj = null;

// Initialize Resolve interface and returns Resolve object.
async function initResolveInterface() {
    // Initialize resolve interface
    const isSuccess = await WorkflowIntegration.Initialize(PLUGIN_ID);
    if (!isSuccess) {
        throw new Error('Error: Failed to initialize Resolve interface!');
        return null;
    }

    // Get resolve interface object
    resolveInterfacObj = await WorkflowIntegration.GetResolve();
    if (!resolveInterfacObj) {
        throw new Error('Error: Failed to get Resolve object!');
        return null;
    }

    return resolveInterfacObj
}

// Cleanup Resolve interface.
function cleanup() {
    const isSuccess = WorkflowIntegration.CleanUp();
    if (!isSuccess) {
        throw new Error('Error: Failed to cleanup Resolve interface!');
    }

    resolveObj = null;
    projectManagerObj = null;
}

// Gets Resolve object.
async function getResolve() {
    if (!resolveObj) {
        resolveObj = await initResolveInterface();
    }

    return resolveObj;
}

// Gets project manager object.
async function getProjectManager() {
    if (!projectManagerObj) {
        resolve = await getResolve();
        if (resolve) {
            projectManagerObj = await resolve.GetProjectManager();
            if (!projectManagerObj) {
                throw new Error('Error: Failed to get ProjectManager object!');
            }
        }
    }

    return projectManagerObj;
}

// Gets current project object.
async function getCurrentProject() {
    curProjManager = await getProjectManager();
    if (curProjManager) {
        currentProject = await curProjManager.GetCurrentProject();
        if (!currentProject) {
            throw new Error('Error: Failed to get current project object!');
        }

        return currentProject;
    }

    return null;
}

// Gets media pool object.
async function getMediaPool() {
    currentProject = await getCurrentProject();
    if (currentProject) {
        mediaPool = await currentProject.GetMediaPool();
        if (!mediaPool) {
            throw new Error('Error: Failed to get MediaPool object!');
        }

        return mediaPool;
    }

    return null;
}

// Gets the current timeline object
async function getCurrentTimeline() {
    currentProject = await getCurrentProject();
    if (currentProject) {
        timeline = await currentProject.GetCurrentTimeline();
        if (!timeline) {
            throw new Error('Error: Failed to get Current Timeline object!')
        }

        return timeline;
    }

    return null;
}

// Gets the current timeline track count
async function getTrackCount(event, trackType) {
    if (typeof trackType !== "string" && trackType) {
        throw new Error("Track Type must be a string!")
        return null;
    } else if (!trackType) {
        throw new Error("Track Type must be provided!")
        return null;
    }
    timeline = await getCurrentTimeline()
    if (timeline) {
        return await timeline.GetTrackCount(trackType);
    }

    return null;
}

async function timelineAddMarker(frame, color, name, note, duration, customData) {
    try {
        const timeline = await getCurrentTimeline();
        if (!timeline) {
            throw new Error('No active timeline');
        }

        const result = await timeline.AddMarker(frame, color, name, note, duration, customData); //"Meta:Krock.io"
        //console.log('AddMarker result:', result);

        return result;
    } catch (error) {
        //console.error('Error in AddMarker:', error);
        throw error;
    }
}

async function timelineDeleteMarker(customData) {
    try {
        const timeline = await getCurrentTimeline();
        if (!timeline) {
            throw new Error('No active timeline');
        }

        const result = await timeline.DeleteMarkerByCustomData(customData);
        //console.log('Markers result:', result);

        return result;
    } catch (error) {
        //console.error('Error in Markers:', error);
        throw error;
    }
}
async function showConfirm(customData) {
    try {
        const timeline = await getCurrentTimeline();
        if (!timeline) {
            throw new Error('No active timeline');
        }

        const result = await timeline.showConfirm(customData);
        //console.log('Markers result:', result);
        return result;
    } catch (error) {
        //console.error('Error in Markers:', error);
        throw error;
    }
}

async function timelineGetMarkers(customData) {
    try {
        const timeline = await getCurrentTimeline();
        if (!timeline) {
            throw new Error('No active timeline');
        }

        const result = await timeline.GetMarkers();
        //console.log('Markers result:', result);

        return result;
    } catch (error) {
        //console.error('Error in Markers:', error);
        throw error;
    }
}

async function setPlayheadTime(timecode) {
    const timeline = await getCurrentTimeline();
    if (timeline) {
        return await timeline.SetCurrentTimecode(timecode);
    }
    return false;
}

function registerResolveEventHandlers() {
    ipcMain.handle('resolve:trackCount', getTrackCount);

    ipcMain.handle('timeline:addMarker', (event, frameId, color, name, note, duration, customData) => 
        timelineAddMarker(frameId, color, name, note, duration, customData));
    ipcMain.handle('timeline:setPlayheadTime', (event, timecode) => setPlayheadTime(timecode));
    ipcMain.handle('timeline:getMarkers', (event, customData) => timelineGetMarkers(customData));
    ipcMain.handle('timeline:deleteMarkerByCustomData', (event, customData) => timelineDeleteMarker(customData));

    ipcMain.handle('show-confirm', async (event, options) => {
        const {
            message,
            detail,
            title = 'Confirm',
            buttons = ['Offset Markers', 'Show First Onl'],
            defaultId = 0,
            cancelId = buttons.length - 1
        } = options;

        const result = await dialog.showMessageBox({
            type: 'question',
            buttons,
            defaultId,
            cancelId,
            title,
            message,
            detail,
            icon: (process.platform === "darwin") ? path.join(__dirname, 'img/logo.icns') : path.join(__dirname, 'img/logo.png'),
        });

        return result.response;
    });
}

const createWindow = () => {
    if (process.platform === "darwin") {
        app.dock.setIcon(path.join(__dirname, 'img/logo.png'));
    }
    // app.dock.setBadge("Krock.io");
    const win = new BrowserWindow ({
        height: 800,
        width: 1200,
        useContentSize: true,
        show: false,
        icon: (process.platform === "darwin") ? path.join(__dirname, 'img/logo.icns') : path.join(__dirname, 'img/logo.png'),
        webPreferences: {
            preload: path.join(__dirname, 'preload.js'),
            nodeIntegration: false,
            contextIsolation: true
        }
    });
    // Show the main window once the content is ready and close the loading window
    win.once('ready-to-show', () => {
        win.show();
        //DevTools
        //win.webContents.openDevTools();
    });

    win.webContents.setWindowOpenHandler(({ url }) => {
        shell.openExternal(url);
        return { action: 'deny' };
    });
    
    // Hide the menu bar (enable below code to hide menu bar)
    //win.setMenu(null);

    win.on('close', function(e) {
        cleanup();
        app.quit();
    });

    // Load index.html on the window.
    win.loadFile('index.html');
};

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.whenReady().then(() => {
    registerResolveEventHandlers();
    createWindow();
});

// Quit when all windows are closed.
app.on('window-all-closed', function () {
    // On macOS it is common for applications and their menu bar
    // to stay active until the user quits explicitly with Cmd + Q
    if (process.platform !== 'darwin') {
        cleanup();
        app.quit();
    }
});

app.on('activate', function () {
    // On macOS it's common to re-create a window in the app when the
    // dock icon is clicked and there are no other windows open.
    if (BrowserWindow.getAllWindows().length === 0) createWindow();
});